import { View, Text, TouchableWithoutFeedback } from "react-native";
import * as Animatable from "react-native-animatable";
import React, { useState } from "react";
import { useRef } from "react";
import Animated, {
    withSpring,
    useAnimatedStyle,
    useSharedValue,
} from "react-native-reanimated";
import { itemDescriptionStyle } from "./ItemDescriptionStyle";
import Description from "./pages/Description";
import Specification from "./pages/Specification";
import Videos from "./pages/Videos";
import { ratingAndReviewsStyle } from "../RatingAndReviews/RatingAndReviewsStyle";
import useLangTranslation from "../../../../../hooks/useLangTranslation";
import { viewDimensions } from "../../../../../helper/utils/viewDimensions";
import usePreferences from "../../../../../hooks/usePreferences";
import { I18nManager } from "react-native";
import dpr from "../../../../Utilities/CustomStyleAttribute/dpr";

const ItemDescription = ({ productDetails, selectedDimensions }) => {
    const { trans } = useLangTranslation();
    const { description, summary, attributes, attribute_values, videos, weight, dimensions } =
		productDetails?.data;
    const descOrSummary = description || summary;
	const { product_general} = usePreferences();
    const [showMore, setShowMore] = useState(false);
    const showMoreRef = useRef(true);
    const [descriptionWidth, setDescriptionWidth] = useState(null);
    const [descriptionHeight, setDescriptionHeight] = useState(null);
    const [specificationWidth, setSpecificationWidth] = useState(null);
    const [videosWidth, setVideosWidth] = useState(null);
    
    const gap1 = useRef(null);
    const gap2 = useRef(null);
    const gap3 = useRef(null);
    const offset = useSharedValue(0);
    const customSpringStyles = useAnimatedStyle(() => {
        return {
            transform: [
                {
                    translateX: withSpring(offset.value, {
                        damping: 20,
                        stiffness: 90,
                    }),
                },
            ],
        };
    });

    let specification = {};
    if (weight) {
        specification = { ...specification, weight: [`${weight} ${product_general?.measurement_weight}`] };
	}
    let specificationLength = Object.keys(specification).length;
    const [tab, setTab] = useState(
        descOrSummary
        ? "description" 
        : specificationLength 
        ? "specification" 
        : "videos"
    );
    
	const isAppearDimensions = (dimensions) => {
		if (dimensions.width || dimensions.height || dimensions.length) {
			return true;
		}
		return false;
	};

	if (
		isAppearDimensions(selectedDimensions) ||
		isAppearDimensions(dimensions)
	) {
		specification = {
			...specification,
			Dimensions: [
                viewDimensions(
                    isAppearDimensions(selectedDimensions) ? selectedDimensions : dimensions,
                    product_general?.measurement_dimension
                ),
			],
		};
	}

    for (let key in attributes) {
        if (attributes[key].visibility == 1) {
            if (attributes[key].attribute_id) {
                specification[key] = [];
                for (let item of attributes[key].value) {
                    for (let k in attribute_values[key]) {
                        if (attribute_values[key][k].id == item) {
                            specification[key].push(
                                attribute_values[key][k].value
                            );
                        }
                    }
                }
            } else if (attributes[key].attribute_id === "") {
                specification[key] = attributes[key].value;
            }
        }
    }

    const [navigation, setNavigation] = useState({
        description: descOrSummary ? true : false,
        specification:
            !descOrSummary && specificationLength > 0 ? true : false,
        videos:
            !descOrSummary && !specification && videos.length > 0 ? true : false,
    });

    const handleDescription = () => {
        setNavigation({
            description: true,
            specification: false,
            videos: false,
        });
        offset.value = 0;
        setTab("description");
    };
    const handleSpecification = () => {
        setNavigation({
            description: false,
            specification: true,
            videos: false,
        });
        offset.value = I18nManager.isRTL ? -gap2.current+videosWidth-descriptionWidth : gap2.current;
        setTab("specification");
    };
    const handleVideos = () => {
        setNavigation({
            description: false,
            specification: false,
            videos: true,
        });
        if(I18nManager.isRTL) {
            if(descOrSummary && specificationLength) {
                offset.value = -gap1.current+dpr(40)
            } else if(descOrSummary && !specificationLength) {
                offset.value = -gap3.current+dpr(18)-descriptionWidth
            } else {
                offset.value = 0
            }
        } else {
            offset.value = gap3.current
        }
        setTab("videos");
    };

    const handleLayout = (event) => {
        const { height } = event.nativeEvent.layout;
        setDescriptionHeight(height);

        if (parseInt(height) > 200) {
            setShowMore(true);
            showMoreRef.current = false;
        }
    };
    const handleShowMore = () => {
        setShowMore(!showMore);
    };
    return (
        <View>
            {(descOrSummary || specificationLength > 0 ||
                videos.length > 0) && (
                <View
                    style={[
                        ratingAndReviewsStyle.subCont,
                        itemDescriptionStyle.subCont,
                    ]}
                >
                    <Animated.View style={customSpringStyles}>
                        {tab === "description" && (
                            <Indicator width={descriptionWidth} />
                        )}
                        {tab === "specification" && (
                            <Indicator width={specificationWidth} />
                        )}
                        {tab === "videos" && <Indicator width={videosWidth} />}
                    </Animated.View>
                    <View style={itemDescriptionStyle.container}>
                        {descOrSummary && (
                            <TouchableWithoutFeedback
                                onLayout={(event) => {
                                    setDescriptionWidth(
                                        event.nativeEvent.layout.width
                                    );
                                    gap1.current = event.nativeEvent.layout.x;
                                }}
                                onPress={handleDescription}
                            >
                                <View>
                                    {descOrSummary && (
                                        <View>
                                            <Text
                                                style={
                                                    itemDescriptionStyle.headerText
                                                }
                                            >
                                                {trans("Description")}
                                            </Text>
                                        </View>
                                    )}
                                </View>
                            </TouchableWithoutFeedback>
                        )}
                        <TouchableWithoutFeedback
                            onLayout={(event) => {
                                setSpecificationWidth(
                                    event.nativeEvent.layout.width
                                );
                                gap2.current = event.nativeEvent.layout.x;
                            }}
                            onPress={handleSpecification}
                        >
                            <View>
                                {specificationLength > 0 && (
                                    <View>
                                        <Text
                                            style={
                                                itemDescriptionStyle.headerText
                                            }
                                        >
                                            {trans("Specification")}
                                        </Text>
                                    </View>
                                )}
                            </View>
                        </TouchableWithoutFeedback>
                        <TouchableWithoutFeedback
                            onLayout={(event) => {
                                setVideosWidth(event.nativeEvent.layout.width);
                                gap3.current = event.nativeEvent.layout.x;
                            }}
                            onPress={handleVideos}
                        >
                            <View>
                                {videos.length > 0 && (
                                    <View>
                                        <Text
                                            style={
                                                itemDescriptionStyle.headerText
                                            }
                                        >
                                            {trans("Videos")}
                                        </Text>
                                    </View>
                                )}
                            </View>
                        </TouchableWithoutFeedback>
                        {!specificationLength > 0 && (
                            <Text style={itemDescriptionStyle.opacityZero}>hidden item</Text>
                        )}
                    </View>
                    <View style={itemDescriptionStyle.descriptionAnimateView}>
                        {descOrSummary && navigation.description && (
                            <View>
                                <Animatable.View
                                    animation='fadeIn'
                                    duration={2000}
                                >
                                    <View
                                        style={
                                            itemDescriptionStyle.boxWithShadow
                                        }
                                        onLayout={
                                            showMoreRef.current && handleLayout
                                        }
                                    >
                                        {descriptionHeight ? (
                                            <Animatable.View
                                                duration={2000}
                                                transition='height'
                                                style={[
                                                    itemDescriptionStyle.description,
                                                    itemDescriptionStyle.dynamicHeight(showMore, descriptionHeight)
                                                ]}
                                            >
                                                <Description
                                                    description={descOrSummary}
                                                />
                                            </Animatable.View>
                                        ) : (
                                            <View>
                                                {showMore ? (
                                                    <View
                                                        style={[
                                                            itemDescriptionStyle.description,
                                                            itemDescriptionStyle.staticHeight
                                                        ]}
                                                    >
                                                        <Description
                                                            description={
                                                                descOrSummary
                                                            }
                                                        />
                                                    </View>
                                                ) : (
                                                    <Description
                                                        description={
                                                            descOrSummary
                                                        }
                                                    />
                                                )}
                                            </View>
                                        )}
                                    </View>
                                    <View>
                                        {!showMoreRef.current && (
                                            <SeeMore
                                                showMore={showMore}
                                                handleShowMore={handleShowMore}
                                            />
                                        )}
                                    </View>
                                </Animatable.View>
                            </View>
                        )}
                        {specificationLength > 0 &&
                            navigation.specification && (
                                <Animatable.View
                                    animation='fadeIn'
                                    duration={2000}
                                >
                                    <Specification
                                        specification={specification}
                                    />
                                </Animatable.View>
                            )}
                        {videos?.length > 0 && navigation?.videos && (
                            <Animatable.View animation='fadeIn' duration={2000}>
                                <Videos videos={videos} />
                            </Animatable.View>
                        )}
                    </View>
                </View>
            )}
        </View>
    );
};

export default React.memo(ItemDescription);

function Indicator({ width }) {
    return (
        <View style={itemDescriptionStyle.positionRelative}>
            <View style={[itemDescriptionStyle.indicator, itemDescriptionStyle.indicatorWidth(width)]} />
        </View>
    );
}

function SeeMore({ showMore, handleShowMore }) {
    const { trans } = useLangTranslation();
    return (
        <TouchableWithoutFeedback onPress={handleShowMore}>
            <View style={itemDescriptionStyle.seeMore}>
                <Text style={itemDescriptionStyle.seeMoreText}>
                    <Text>{showMore ? trans("See More") : trans("See Less")}</Text>
                </Text>
            </View>
        </TouchableWithoutFeedback>
    );
}
